# Plugin Structure Documentation

## Directory Structure

```
internal-link-suggester/
│
├── internal-link-suggester.php    # Main plugin file, handles initialization
├── uninstall.php                   # Cleanup script when plugin is deleted
├── index.php                       # Security file (prevents directory browsing)
│
├── README.md                       # Comprehensive documentation
├── INSTALLATION.md                 # Installation instructions
├── QUICKSTART.md                   # Quick start guide
├── FEATURES.md                     # Feature overview
├── STRUCTURE.md                    # This file
├── .gitignore                      # Git ignore rules
│
├── includes/                       # Core plugin classes
│   ├── class-sitemap-parser.php       # Parses sitemap.xml files
│   ├── class-keyword-matcher.php      # Matches keywords in content
│   ├── class-link-suggester.php       # Main suggestion engine
│   └── index.php                      # Security file
│
├── admin/                          # Admin interface files
│   ├── class-admin-interface.php      # Admin pages and forms
│   └── index.php                      # Security file
│
└── assets/                         # Frontend assets
    ├── index.php                      # Security file
    │
    ├── css/                           # Stylesheets
    │   ├── admin-style.css               # Admin CSS
    │   └── index.php                     # Security file
    │
    └── js/                            # JavaScript files
        ├── admin-script.js               # Admin JavaScript
        ├── editor-script.js              # Gutenberg integration
        └── index.php                     # Security file
```

## File Descriptions

### Root Level Files

#### internal-link-suggester.php
**Purpose**: Main plugin file
**Responsibilities**:
- Plugin header with metadata
- Plugin constants definition
- File includes
- Main plugin class
- Hook initialization
- Database table creation
- AJAX handlers
- Meta box registration
- Cron job scheduling

**Key Functions**:
- `Internal_Link_Suggester::get_instance()` - Singleton instance
- `activate()` - Plugin activation
- `deactivate()` - Plugin deactivation
- `enqueue_admin_assets()` - Load admin scripts
- `enqueue_editor_assets()` - Load editor scripts
- `ajax_get_suggestions()` - AJAX handler for suggestions
- `ajax_refresh_sitemap()` - AJAX handler for sitemap refresh
- `ajax_save_keywords()` - AJAX handler for saving keywords

#### uninstall.php
**Purpose**: Clean uninstallation
**Responsibilities**:
- Delete plugin options
- Remove scheduled cron jobs
- Drop database table
- Clear caches

### Includes Directory

#### class-sitemap-parser.php
**Purpose**: Sitemap XML parsing
**Class**: `ILS_Sitemap_Parser`

**Key Methods**:
- `parse($sitemap_url)` - Parse sitemap and extract URLs
- `extract_title($url, $namespaces, $loc)` - Extract page title
- `generate_title_from_url($url)` - Generate title from URL slug
- `validate_sitemap_url($url)` - Validate sitemap URL

**Features**:
- Handles sitemap index files
- Supports multiple namespaces
- Extracts titles from various sources
- Recursive parsing for nested sitemaps
- Error handling and logging

#### class-keyword-matcher.php
**Purpose**: Keyword matching logic
**Class**: `ILS_Keyword_Matcher`

**Key Methods**:
- `find_matches($content, $links, $exclude_post_id)` - Find keyword matches in content
- `find_best_match_for_text($selected_text, $links)` - Find best match for specific text
- `suggest_keywords_from_content($content, $max_suggestions)` - Auto-suggest keywords
- `parse_keywords($keywords)` - Parse keyword string to array
- `extract_existing_links($content)` - Extract existing links from content

**Features**:
- Case-insensitive matching
- Frequency-based scoring
- Duplicate link prevention
- Partial and exact matching
- Stop word filtering

#### class-link-suggester.php
**Purpose**: Main suggestion engine
**Class**: `ILS_Link_Suggester`

**Key Methods**:
- `get_suggestions($content, $exclude_post_id)` - Get link suggestions for content
- `get_suggestion_for_text($text)` - Get suggestion for specific text
- `suggest_keywords_for_url($url)` - Suggest keywords for a URL
- `get_statistics()` - Get plugin statistics
- `get_all_links()` - Retrieve all links from database

**Features**:
- Integrates parser and matcher
- Manages database queries
- Provides statistics
- Limits results appropriately

### Admin Directory

#### class-admin-interface.php
**Purpose**: WordPress admin interface
**Class**: `ILS_Admin_Interface`

**Key Methods**:
- `add_admin_menu()` - Register admin menu pages
- `register_settings()` - Register plugin settings
- `render_main_page()` - Dashboard page
- `render_keywords_page()` - Keywords management page
- `render_settings_page()` - Settings page

**Admin Pages**:
1. **Dashboard** - Statistics and quick actions
2. **Manage Keywords** - Keyword management with search and pagination
3. **Settings** - Plugin configuration

**Features**:
- Statistics display
- Bulk operations
- Search functionality
- Pagination
- Form handling

### Assets Directory

#### css/admin-style.css
**Purpose**: Admin interface styling

**Key Styles**:
- Dashboard stats grid
- Action buttons
- Keywords table
- Search box
- Meta box styling
- Responsive design
- Loading states

**Design Features**:
- Clean, modern look
- WordPress admin consistency
- Mobile responsive
- Accessible colors
- Smooth transitions

#### js/admin-script.js
**Purpose**: Admin functionality

**Key Features**:
- Sitemap refresh AJAX
- Keyword save AJAX
- Bulk save operations
- Link copying
- Visual feedback
- Loading states

**Dependencies**:
- jQuery
- WordPress AJAX API

#### js/editor-script.js
**Purpose**: Gutenberg editor integration

**Key Features**:
- Sidebar panel registration
- Real-time suggestions
- Auto-refresh on content change
- Link copying
- React-based UI

**Dependencies**:
- WordPress Gutenberg APIs
- React (via WordPress)

## Database Schema

### Table: wp_ils_links

```sql
CREATE TABLE wp_ils_links (
    id mediumint(9) NOT NULL AUTO_INCREMENT,
    url varchar(500) NOT NULL,
    title varchar(255) DEFAULT '',
    keywords text DEFAULT '',
    last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    UNIQUE KEY url (url)
);
```

**Fields**:
- `id` - Unique identifier
- `url` - Page URL (unique)
- `title` - Page title
- `keywords` - Comma-separated keywords
- `last_updated` - Auto-updated timestamp

**Indexes**:
- Primary key on `id`
- Unique index on `url`

## WordPress Options

### Plugin Settings

- `ils_sitemap_url` - Sitemap URL (default: site_url/sitemap.xml)
- `ils_max_suggestions` - Maximum suggestions per post (default: 5)
- `ils_auto_refresh` - Auto-refresh enabled (default: 1)

## WordPress Hooks

### Actions Used

- `admin_menu` - Register admin pages
- `admin_init` - Register settings
- `admin_enqueue_scripts` - Enqueue admin assets
- `enqueue_block_editor_assets` - Enqueue editor assets
- `add_meta_boxes` - Add meta box
- `ils_refresh_sitemap_cron` - Cron job for sitemap refresh

### AJAX Actions

- `wp_ajax_ils_get_suggestions` - Get link suggestions
- `wp_ajax_ils_refresh_sitemap` - Refresh sitemap
- `wp_ajax_ils_save_keywords` - Save keywords

### Activation/Deactivation

- `register_activation_hook` - Create table, set defaults
- `register_deactivation_hook` - Clear cron jobs

## Security Measures

### Input Validation
- URL validation with `esc_url_raw()`
- Text sanitization with `sanitize_text_field()`
- Integer validation with `intval()`

### Output Escaping
- HTML output with `esc_html()`
- Attribute output with `esc_attr()`
- URL output with `esc_url()`

### Authentication
- Nonce verification for AJAX
- Capability checks for admin
- Direct access prevention

### Database Security
- Prepared statements
- Escaped SQL queries
- Proper table prefix usage

## Data Flow

### Sitemap Import Flow
1. User clicks "Refresh Sitemap"
2. JavaScript sends AJAX request
3. PHP fetches sitemap XML
4. Parser extracts URLs and titles
5. Database is updated/inserted
6. Response sent back to JavaScript
7. UI updated with results

### Suggestion Generation Flow
1. User writes content in editor
2. JavaScript detects content change
3. AJAX request sent with content
4. Keyword matcher analyzes content
5. Database queried for links
6. Matches found and ranked
7. Top suggestions returned
8. UI displays suggestions

### Keyword Save Flow
1. User enters keywords in textarea
2. User clicks Save button
3. JavaScript sends AJAX request
4. Server validates and sanitizes
5. Database updated
6. Success response returned
7. UI shows confirmation

## Performance Considerations

### Optimization Techniques
- Database caching
- Efficient SQL queries
- Conditional script loading
- Debounced content analysis
- Pagination for large datasets

### Resource Usage
- Minimal front-end impact (zero)
- Background cron processing
- Efficient keyword matching
- Optimized database queries

## Extension Points

### Potential Customizations
- Add custom post types
- Filter suggestions
- Modify matching algorithm
- Add custom admin pages
- Integrate with other plugins

### Developer Hooks (Future)
- Filter: `ils_suggestion_results`
- Filter: `ils_keyword_match_score`
- Action: `ils_after_sitemap_refresh`
- Action: `ils_after_keyword_save`

## Testing Checklist

### Installation Testing
- [ ] Plugin activates without errors
- [ ] Database table created
- [ ] Default options set
- [ ] Admin menu appears

### Functionality Testing
- [ ] Sitemap refresh works
- [ ] Keywords save correctly
- [ ] Suggestions appear in editor
- [ ] Search works in keywords page
- [ ] Settings save correctly

### Compatibility Testing
- [ ] Works with Gutenberg
- [ ] Works with Classic Editor
- [ ] Compatible with common themes
- [ ] No JavaScript conflicts
- [ ] Mobile responsive

## Maintenance

### Regular Tasks
- Update documentation
- Test with new WordPress versions
- Review and optimize queries
- Update dependencies
- Security audits

### Monitoring
- Check error logs
- Monitor database size
- Track performance
- User feedback
- Bug reports

## Version History

**1.0.0** - Initial Release
- Core functionality
- Admin interface
- Editor integration
- Documentation
