<?php
/**
 * Admin Interface Class
 * Handles the WordPress admin interface for SEOINUX
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Admin_Interface {

    private $database;

    public function __construct() {
        $this->database = new SEOINUX_Database();
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        add_menu_page(
            'SEOINUX Link Creator',
            'Link Creator',
            'manage_options',
            'seoinux-link-creator',
            array($this, 'render_main_page'),
            'dashicons-admin-links',
            30
        );

        add_submenu_page(
            'seoinux-link-creator',
            'Manage Keywords',
            'Manage Keywords',
            'manage_options',
            'seoinux-manage-keywords',
            array($this, 'render_keywords_page')
        );

        add_submenu_page(
            'seoinux-link-creator',
            'Settings',
            'Settings',
            'manage_options',
            'seoinux-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('seoinux_settings', 'seoinux_sitemap_url');
        register_setting('seoinux_settings', 'seoinux_max_suggestions');
        register_setting('seoinux_settings', 'seoinux_auto_refresh');
        register_setting('seoinux_settings', 'seoinux_auto_generate_keywords');
    }

    /**
     * Render main dashboard page
     */
    public function render_main_page() {
        $suggester = new SEOINUX_Link_Suggester();
        $stats = $suggester->get_statistics();

        ?>
        <div class="wrap seoinux-admin-wrap">
            <div class="seoinux-header">
                <div class="seoinux-branding">
                    <img src="https://seoinux.com/wp-content/uploads/2025/01/main-logo-seo.png" alt="SEOINUX" class="seoinux-logo">
                    <div class="seoinux-title-group">
                        <h1>SEO Internal Link Creator</h1>
                        <p class="description">Auto-generates keywords from titles</p>
                    </div>
                </div>
            </div>

            <div class="seoinux-dashboard">
                <div class="seoinux-stats-grid">
                    <div class="seoinux-stat-card">
                        <div class="seoinux-stat-icon">
                            <span class="dashicons dashicons-admin-links"></span>
                        </div>
                        <div class="seoinux-stat-content">
                            <h3>Total Links</h3>
                            <p class="seoinux-stat-number"><?php echo esc_html($stats['total_links']); ?></p>
                        </div>
                    </div>

                    <div class="seoinux-stat-card">
                        <div class="seoinux-stat-icon seoinux-stat-icon-custom">
                            <span class="dashicons dashicons-edit"></span>
                        </div>
                        <div class="seoinux-stat-content">
                            <h3>Custom Keywords</h3>
                            <p class="seoinux-stat-number"><?php echo esc_html($stats['with_custom_keywords']); ?></p>
                        </div>
                    </div>

                    <div class="seoinux-stat-card">
                        <div class="seoinux-stat-icon seoinux-stat-icon-auto">
                            <span class="dashicons dashicons-update"></span>
                        </div>
                        <div class="seoinux-stat-content">
                            <h3>Auto Keywords</h3>
                            <p class="seoinux-stat-number"><?php echo esc_html($stats['with_auto_keywords']); ?></p>
                        </div>
                    </div>
                </div>

                <div class="seoinux-actions-section">
                    <h2>Quick Actions</h2>
                    <div class="seoinux-button-group">
                        <button type="button" class="button button-primary button-large" id="seoinux-refresh-sitemap">
                            <span class="dashicons dashicons-update"></span> Refresh Sitemap & Auto-Generate Keywords
                        </button>
                        <a href="<?php echo admin_url('admin.php?page=seoinux-manage-keywords'); ?>" class="button button-secondary button-large">
                            <span class="dashicons dashicons-edit"></span> Manage Keywords
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=seoinux-settings'); ?>" class="button button-secondary button-large">
                            <span class="dashicons dashicons-admin-settings"></span> Settings
                        </a>
                    </div>
                    <div id="seoinux-refresh-message"></div>
                </div>

                <div class="seoinux-two-column">
                    <div class="seoinux-info-section">
                        <h2>How It Works</h2>
                        <ol>
                            <li><strong>Refresh Sitemap:</strong> Fetches all URLs and auto-generates keywords from page titles</li>
                            <li><strong>Edit Keywords:</strong> Review auto-generated keywords and add custom ones</li>
                            <li><strong>Write Content:</strong> Get intelligent link suggestions based on your content</li>
                            <li><strong>Insert Links:</strong> Easily copy and paste suggested links into your articles</li>
                        </ol>
                        <div class="seoinux-feature-badge">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <strong>New:</strong> Keywords are automatically generated from post/page titles!
                        </div>
                    </div>

                    <div class="seoinux-tools-promo">
                        <div class="seoinux-tools-header">
                            <img src="https://seoinux.com/wp-content/uploads/2025/01/favicon.png" alt="SEOINUX Tools" class="seoinux-tools-icon">
                            <h2>Totally Free SEO Reports & Tools</h2>
                        </div>
                        <p class="seoinux-tools-tagline"><strong>Identify technical SEO issues and take action to improve the health and performance of your website.</strong></p>
                        <ul class="seoinux-tools-features">
                            <li><span class="dashicons dashicons-chart-line"></span> Advanced SEO reports with AI enhancement</li>
                            <li><span class="dashicons dashicons-search"></span> Comprehensive SEO analysis and audits</li>
                            <li><span class="dashicons dashicons-admin-tools"></span> Over 30+ professional web tools and utilities</li>
                            <li><span class="dashicons dashicons-awards"></span> Ranking improvement on major search engines</li>
                        </ul>
                        <a href="https://tools.seoinux.com/" target="_blank" class="button button-primary button-large seoinux-tools-cta">
                            <span class="dashicons dashicons-external"></span> Access Free SEO Tools
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render keywords management page
     */
    public function render_keywords_page() {
        // Pagination
        $per_page = 20;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;

        // Search
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';

        if (!empty($search)) {
            $all_links = $this->database->search_links($search);
            $total_items = count($all_links);
            $links = array_slice($all_links, $offset, $per_page);
        } else {
            $all_links = $this->database->get_all_links();
            $total_items = count($all_links);
            $links = array_slice($all_links, $offset, $per_page);
        }

        $total_pages = ceil($total_items / $per_page);
        $keyword_generator = new SEOINUX_Keyword_Generator();

        ?>
        <div class="wrap seoinux-admin-wrap">
            <h1>Manage Keywords</h1>
            <p class="description">Auto-generated keywords from titles are shown. You can edit them or add custom keywords.</p>

            <div class="seoinux-search-box">
                <form method="get">
                    <input type="hidden" name="page" value="seoinux-manage-keywords">
                    <input type="search" name="s" value="<?php echo esc_attr($search); ?>" placeholder="Search URLs, titles, or keywords...">
                    <button type="submit" class="button">Search</button>
                </form>
            </div>

            <?php if (empty($links)): ?>
                <div class="seoinux-notice">
                    <p>No links found. Please refresh your sitemap from the main dashboard.</p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped seoinux-keywords-table">
                    <thead>
                        <tr>
                            <th style="width: 35%">URL / Title</th>
                            <th style="width: 25%">Auto-Generated Keywords</th>
                            <th style="width: 30%">Custom Keywords (editable)</th>
                            <th style="width: 10%">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($links as $link): ?>
                            <tr data-url="<?php echo esc_attr($link['url']); ?>">
                                <td>
                                    <strong><?php echo esc_html($link['title']); ?></strong><br>
                                    <small><a href="<?php echo esc_url($link['url']); ?>" target="_blank"><?php echo esc_html($link['url']); ?></a></small>
                                </td>
                                <td>
                                    <div class="seoinux-auto-keywords">
                                        <?php
                                        $auto_kw = !empty($link['auto_keywords']) ? $link['auto_keywords'] : '';
                                        if (empty($auto_kw)) {
                                            echo '<em style="color: #999;">None generated</em>';
                                        } else {
                                            echo '<small>' . esc_html($auto_kw) . '</small>';
                                        }
                                        ?>
                                    </div>
                                    <button type="button" class="button button-small seoinux-regenerate-keywords" data-url="<?php echo esc_attr($link['url']); ?>" style="margin-top: 5px;">
                                        Regenerate
                                    </button>
                                </td>
                                <td>
                                    <textarea
                                        class="seoinux-keywords-input"
                                        rows="3"
                                        placeholder="Add custom keywords (comma-separated)"
                                        data-url="<?php echo esc_attr($link['url']); ?>"
                                    ><?php echo esc_textarea($link['keywords'] ?? ''); ?></textarea>
                                </td>
                                <td>
                                    <button type="button" class="button button-primary seoinux-save-keywords" data-url="<?php echo esc_attr($link['url']); ?>">
                                        Save
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php if ($total_pages > 1): ?>
                    <div class="tablenav">
                        <div class="tablenav-pages">
                            <?php
                            $base_url = add_query_arg('page', 'seoinux-manage-keywords', admin_url('admin.php'));
                            if (!empty($search)) {
                                $base_url = add_query_arg('s', $search, $base_url);
                            }

                            echo paginate_links(array(
                                'base' => add_query_arg('paged', '%#%', $base_url),
                                'format' => '',
                                'prev_text' => '&laquo;',
                                'next_text' => '&raquo;',
                                'total' => $total_pages,
                                'current' => $current_page
                            ));
                            ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="seoinux-bulk-actions">
                    <button type="button" class="button button-primary button-large" id="seoinux-save-all-keywords">
                        Save All Custom Keywords
                    </button>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (isset($_POST['seoinux_save_settings'])) {
            check_admin_referer('seoinux_settings_nonce');

            update_option('seoinux_sitemap_url', esc_url_raw($_POST['seoinux_sitemap_url']));
            update_option('seoinux_max_suggestions', intval($_POST['seoinux_max_suggestions']));
            update_option('seoinux_auto_refresh', isset($_POST['seoinux_auto_refresh']) ? 1 : 0);
            update_option('seoinux_auto_generate_keywords', isset($_POST['seoinux_auto_generate_keywords']) ? 1 : 0);

            echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
        }

        $sitemap_url = get_option('seoinux_sitemap_url', site_url('/sitemap.xml'));
        $max_suggestions = get_option('seoinux_max_suggestions', 5);
        $auto_refresh = get_option('seoinux_auto_refresh', 1);
        $auto_generate = get_option('seoinux_auto_generate_keywords', 1);

        ?>
        <div class="wrap seoinux-admin-wrap">
            <h1>SEOINUX Settings</h1>

            <form method="post" action="">
                <?php wp_nonce_field('seoinux_settings_nonce'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="seoinux_sitemap_url">Sitemap URL</label>
                        </th>
                        <td>
                            <input
                                type="url"
                                name="seoinux_sitemap_url"
                                id="seoinux_sitemap_url"
                                value="<?php echo esc_attr($sitemap_url); ?>"
                                class="regular-text"
                                required
                            >
                            <p class="description">Enter the full URL to your sitemap.xml file.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="seoinux_max_suggestions">Maximum Suggestions</label>
                        </th>
                        <td>
                            <input
                                type="number"
                                name="seoinux_max_suggestions"
                                id="seoinux_max_suggestions"
                                value="<?php echo esc_attr($max_suggestions); ?>"
                                min="1"
                                max="20"
                                required
                            >
                            <p class="description">Maximum number of link suggestions to display per post.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="seoinux_auto_generate_keywords">Auto-Generate Keywords</label>
                        </th>
                        <td>
                            <label>
                                <input
                                    type="checkbox"
                                    name="seoinux_auto_generate_keywords"
                                    id="seoinux_auto_generate_keywords"
                                    value="1"
                                    <?php checked($auto_generate, 1); ?>
                                >
                                Automatically generate keywords from page titles
                            </label>
                            <p class="description">When enabled, keywords are automatically extracted from page/post titles during sitemap refresh.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="seoinux_auto_refresh">Auto-refresh Sitemap</label>
                        </th>
                        <td>
                            <label>
                                <input
                                    type="checkbox"
                                    name="seoinux_auto_refresh"
                                    id="seoinux_auto_refresh"
                                    value="1"
                                    <?php checked($auto_refresh, 1); ?>
                                >
                                Automatically refresh sitemap daily
                            </label>
                            <p class="description">When enabled, the sitemap will be refreshed automatically once per day.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            Database
                        </th>
                        <td>
                            <p><strong>Status:</strong> ✅ WordPress Database</p>
                            <p class="description">All keywords are stored in your WordPress database.</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="seoinux_save_settings" class="button button-primary" value="Save Settings">
                </p>
            </form>
        </div>
        <?php
    }
}
