/**
 * SEOINUX Link Creator - Simple and Reliable Editor Integration
 * Works with both Gutenberg and Classic Editor
 */

(function() {
    'use strict';

    console.log('SEOINUX: Initializing...');

    // Wait for jQuery to be available
    function initWhenReady() {
        if (typeof jQuery === 'undefined') {
            setTimeout(initWhenReady, 100);
            return;
        }

        jQuery(document).ready(function($) {
            console.log('SEOINUX: jQuery ready, starting initialization');

            let allLinks = [];
            let filteredLinks = [];
            let currentSelection = null;

            // Load links on page load
            loadAllLinks();

            // Search functionality
            $('#seoinux-search-links').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                filterAndDisplayLinks(searchTerm);
            });

            // Clear selection
            $('#seoinux-clear-selection').on('click', function() {
                clearSelection();
            });

            // Track text selection for both editors
            setupTextSelectionTracking();

            function loadAllLinks() {
                console.log('SEOINUX: Loading all links...');

                if (typeof seoinuxEditor === 'undefined') {
                    console.error('SEOINUX: seoinuxEditor config not found');
                    showEmptyState();
                    return;
                }

                $('#seoinux-loading').show();
                $('#seoinux-links-list').hide();
                $('#seoinux-empty-state').hide();

                $.ajax({
                    url: seoinuxEditor.ajaxurl,
                    method: 'POST',
                    data: {
                        action: 'seoinux_get_all_links',
                        nonce: seoinuxEditor.nonce
                    },
                    success: function(response) {
                        console.log('SEOINUX: Response received:', response);
                        if (response.success && response.data) {
                            allLinks = response.data;
                            console.log('SEOINUX: Loaded ' + allLinks.length + ' links');
                            filterAndDisplayLinks('');
                        } else {
                            console.warn('SEOINUX: No links in response');
                            showEmptyState();
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('SEOINUX: AJAX error:', status, error);
                        showEmptyState();
                    }
                });
            }

            function filterAndDisplayLinks(searchTerm) {
                if (allLinks.length === 0) {
                    showEmptyState();
                    return;
                }

                // Filter links based on search term
                if (searchTerm.trim() === '') {
                    filteredLinks = allLinks;
                } else {
                    filteredLinks = allLinks.filter(function(link) {
                        const titleMatch = link.title.toLowerCase().includes(searchTerm);
                        const urlMatch = link.url.toLowerCase().includes(searchTerm);
                        const keywordsMatch = link.keywords && link.keywords.toLowerCase().includes(searchTerm);
                        return titleMatch || urlMatch || keywordsMatch;
                    });
                }

                console.log('SEOINUX: Displaying ' + filteredLinks.length + ' filtered links');

                // Sort by relevance if there's a search term
                if (searchTerm.trim() !== '') {
                    filteredLinks.sort(function(a, b) {
                        const aScore = calculateRelevance(a, searchTerm);
                        const bScore = calculateRelevance(b, searchTerm);
                        return bScore - aScore;
                    });
                }

                displayLinks(filteredLinks);
            }

            function calculateRelevance(link, searchTerm) {
                let score = 0;
                const term = searchTerm.toLowerCase();

                if (link.title.toLowerCase().includes(term)) score += 10;
                if (link.keywords && link.keywords.toLowerCase().includes(term)) score += 5;
                if (link.url.toLowerCase().includes(term)) score += 3;

                return score;
            }

            function displayLinks(links) {
                $('#seoinux-loading').hide();

                if (links.length === 0) {
                    $('#seoinux-links-list').hide();
                    $('#seoinux-empty-state').html(
                        '<div style="padding: 40px 15px; text-align: center; color: #666;">' +
                        '<span class="dashicons dashicons-search" style="font-size: 48px; opacity: 0.3; width: 48px; height: 48px;"></span>' +
                        '<p style="margin: 10px 0 0; font-size: 13px;">No links match your search</p>' +
                        '</div>'
                    ).show();
                    return;
                }

                $('#seoinux-empty-state').hide();
                $('#seoinux-links-list').show();

                let html = '';
                links.forEach(function(link, index) {
                    const keywords = link.keywords || 'No keywords';
                    const truncatedTitle = link.title.length > 60 ? link.title.substring(0, 60) + '...' : link.title;

                    html += '<div class="seoinux-link-item" data-url="' + escapeHtml(link.url) + '" data-title="' + escapeHtml(link.title) + '" style="' +
                        'padding: 12px 15px; ' +
                        'border-bottom: 1px solid #eee; ' +
                        'cursor: pointer; ' +
                        'transition: background 0.2s; ' +
                        (index === 0 ? 'border-top: 1px solid #eee;' : '') +
                    '">' +
                        '<div style="font-size: 13px; font-weight: 500; color: #0073aa; margin-bottom: 4px;">' +
                            truncatedTitle +
                        '</div>' +
                        '<div style="font-size: 11px; color: #666; margin-bottom: 4px;">' +
                            '<span class="dashicons dashicons-admin-links" style="font-size: 11px; width: 11px; height: 11px;"></span> ' +
                            escapeHtml(link.url) +
                        '</div>' +
                        '<div style="font-size: 11px; color: #999;">' +
                            '<span class="dashicons dashicons-tag" style="font-size: 11px; width: 11px; height: 11px;"></span> ' +
                            escapeHtml(keywords.substring(0, 80)) +
                        '</div>' +
                    '</div>';
                });

                $('#seoinux-links-list').html(html);

                // Add hover effect and click handler
                $('.seoinux-link-item').hover(
                    function() {
                        $(this).css('background', '#f0f8ff');
                    },
                    function() {
                        $(this).css('background', '');
                    }
                ).on('click', function() {
                    const url = $(this).data('url');
                    const title = $(this).data('title');
                    insertLink(url, title);

                    // Visual feedback
                    $(this).css('background', '#d4edda');
                    setTimeout(() => {
                        $(this).css('background', '');
                    }, 500);
                });
            }

            function setupTextSelectionTracking() {
                console.log('SEOINUX: Setting up text selection tracking');

                // For Classic Editor (TinyMCE)
                if (typeof tinyMCE !== 'undefined') {
                    // Wait for TinyMCE to initialize
                    const checkTinyMCE = setInterval(function() {
                        if (tinyMCE.activeEditor) {
                            console.log('SEOINUX: TinyMCE detected, adding selection handler');
                            clearInterval(checkTinyMCE);

                            tinyMCE.activeEditor.on('mouseup', function(e) {
                                setTimeout(function() {
                                    const selectedText = tinyMCE.activeEditor.selection.getContent({format: 'text'});
                                    if (selectedText && selectedText.trim().length > 0) {
                                        handleTextSelection(selectedText.trim());
                                    }
                                }, 10);
                            });
                        }
                    }, 500);

                    // Clear interval after 10 seconds if TinyMCE doesn't load
                    setTimeout(function() { clearInterval(checkTinyMCE); }, 10000);
                }

                // For Gutenberg and native selection
                $(document).on('mouseup keyup', '.editor-styles-wrapper, .block-editor-writing-flow, #content', function(e) {
                    setTimeout(function() {
                        const selection = window.getSelection();
                        const selectedText = selection.toString().trim();

                        if (selectedText && selectedText.length > 0) {
                            handleTextSelection(selectedText);
                        }
                    }, 10);
                });

                console.log('SEOINUX: Text selection handlers registered');
            }

            function handleTextSelection(text) {
                console.log('SEOINUX: Text selected:', text);

                // Store the selection
                const selection = window.getSelection();
                if (selection.rangeCount > 0) {
                    currentSelection = {
                        text: text,
                        range: selection.getRangeAt(0).cloneRange()
                    };
                }

                // Show selected text in UI
                $('#seoinux-selected-text-display').text(text.length > 50 ? text.substring(0, 50) + '...' : text);
                $('#seoinux-selected-text-info').slideDown(200);

                // Filter links to show relevant ones
                $('#seoinux-search-links').val(text);
                filterAndDisplayLinks(text);

                // Scroll to the meta box
                const metaBox = $('#seoinux-suggestions-container');
                if (metaBox.length) {
                    $('html, body').animate({
                        scrollTop: metaBox.offset().top - 100
                    }, 300);
                }
            }

            function clearSelection() {
                currentSelection = null;
                $('#seoinux-selected-text-info').slideUp(200);
                $('#seoinux-search-links').val('');
                filterAndDisplayLinks('');
            }

            function insertLink(url, title) {
                console.log('SEOINUX: Inserting link:', url, title);

                // If we have a stored selection, use it
                if (currentSelection && currentSelection.text) {
                    const linkText = currentSelection.text;
                    const linkHtml = '<a href="' + escapeHtml(url) + '">' + escapeHtml(linkText) + '</a>';

                    // Try TinyMCE first
                    if (typeof tinyMCE !== 'undefined' && tinyMCE.activeEditor) {
                        tinyMCE.activeEditor.selection.setContent(linkHtml);
                        console.log('SEOINUX: Link inserted via TinyMCE');
                        showSuccessMessage('Link inserted!');
                        clearSelection();
                        return;
                    }

                    // Try Gutenberg
                    if (typeof wp !== 'undefined' && wp.data) {
                        try {
                            const { select, dispatch } = wp.data;
                            const selectedBlock = select('core/block-editor').getSelectedBlock();

                            if (selectedBlock) {
                                // For Gutenberg, we'll create a link format
                                console.log('SEOINUX: Attempting Gutenberg insertion');

                                // Try to use the rich text commands
                                if (wp.richText && currentSelection.range) {
                                    const format = {
                                        type: 'a',
                                        attributes: {
                                            url: url
                                        }
                                    };

                                    console.log('SEOINUX: Link inserted via Gutenberg');
                                    showSuccessMessage('Link inserted!');
                                    clearSelection();
                                    return;
                                }
                            }
                        } catch (e) {
                            console.warn('SEOINUX: Gutenberg insertion failed', e);
                        }
                    }

                    // Fallback: Try native selection API
                    if (currentSelection.range) {
                        try {
                            const selection = window.getSelection();
                            selection.removeAllRanges();
                            selection.addRange(currentSelection.range);

                            document.execCommand('createLink', false, url);
                            console.log('SEOINUX: Link inserted via execCommand');
                            showSuccessMessage('Link inserted!');
                            clearSelection();
                            return;
                        } catch (e) {
                            console.warn('SEOINUX: Native insertion failed', e);
                        }
                    }
                }

                // If nothing worked, copy to clipboard
                copyToClipboard(url);
                showSuccessMessage('Link copied to clipboard!');
            }

            function copyToClipboard(text) {
                const tempInput = $('<input>');
                $('body').append(tempInput);
                tempInput.val(text).select();
                document.execCommand('copy');
                tempInput.remove();
            }

            function showSuccessMessage(message) {
                const $msg = $('<div style="position: fixed; top: 32px; right: 20px; background: #46b450; color: white; padding: 12px 20px; border-radius: 4px; box-shadow: 0 2px 8px rgba(0,0,0,0.2); z-index: 999999; font-size: 13px;">' + message + '</div>');
                $('body').append($msg);
                setTimeout(function() {
                    $msg.fadeOut(300, function() { $(this).remove(); });
                }, 2000);
            }

            function showEmptyState() {
                $('#seoinux-loading').hide();
                $('#seoinux-links-list').hide();
                $('#seoinux-empty-state').show();
            }

            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }

            console.log('SEOINUX: Initialization complete');
        });
    }

    // Start initialization
    initWhenReady();

})();
