<?php
if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Database {

    private $links_table;

    public function __construct() {
        global $wpdb;
        $this->links_table = $wpdb->prefix . 'seoinux_links';
    }

    public function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS {$this->links_table} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            url varchar(500) NOT NULL,
            title text NOT NULL,
            keywords text DEFAULT '',
            auto_keywords text DEFAULT '',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY url (url),
            KEY keywords_index (keywords(191)),
            KEY auto_keywords_index (auto_keywords(191))
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        error_log('SEOINUX: Database tables created successfully');
    }

    public function upsert_link($url, $title, $auto_keywords = '', $keywords = null) {
        global $wpdb;

        $existing = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->links_table} WHERE url = %s",
                $url
            ),
            ARRAY_A
        );

        $data = array(
            'url' => $url,
            'title' => $title,
            'auto_keywords' => $auto_keywords,
            'updated_at' => current_time('mysql')
        );

        if ($keywords !== null) {
            $data['keywords'] = $keywords;
        } elseif (!$existing) {
            $data['keywords'] = '';
        }

        if ($existing) {
            if ($keywords === null) {
                unset($data['keywords']);
            }

            $result = $wpdb->update(
                $this->links_table,
                $data,
                array('url' => $url),
                array('%s', '%s', '%s', '%s'),
                array('%s')
            );

            return $result !== false ? array('id' => $existing['id']) : array('error' => $wpdb->last_error);
        } else {
            $data['created_at'] = current_time('mysql');

            $result = $wpdb->insert(
                $this->links_table,
                $data,
                array('%s', '%s', '%s', '%s', '%s', '%s')
            );

            return $result !== false ? array('id' => $wpdb->insert_id) : array('error' => $wpdb->last_error);
        }
    }

    public function get_all_links() {
        global $wpdb;

        $results = $wpdb->get_results(
            "SELECT * FROM {$this->links_table} ORDER BY updated_at DESC",
            ARRAY_A
        );

        return $results !== null ? $results : array();
    }

    public function get_link_by_url($url) {
        global $wpdb;

        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->links_table} WHERE url = %s",
                $url
            ),
            ARRAY_A
        );

        return $result !== null ? $result : array('error' => 'Link not found');
    }

    public function update_keywords($url, $keywords) {
        global $wpdb;

        $result = $wpdb->update(
            $this->links_table,
            array(
                'keywords' => $keywords,
                'updated_at' => current_time('mysql')
            ),
            array('url' => $url),
            array('%s', '%s'),
            array('%s')
        );

        return $result !== false ? array('success' => true) : array('error' => $wpdb->last_error);
    }

    public function delete_link($url) {
        global $wpdb;

        $result = $wpdb->delete(
            $this->links_table,
            array('url' => $url),
            array('%s')
        );

        return $result !== false ? array('success' => true) : array('error' => $wpdb->last_error);
    }

    public function search_links($keyword) {
        global $wpdb;

        $like = '%' . $wpdb->esc_like($keyword) . '%';

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$this->links_table}
                WHERE keywords LIKE %s
                OR auto_keywords LIKE %s
                OR title LIKE %s
                ORDER BY updated_at DESC",
                $like,
                $like,
                $like
            ),
            ARRAY_A
        );

        return $results !== null ? $results : array();
    }

    public function drop_tables() {
        global $wpdb;
        $wpdb->query("DROP TABLE IF EXISTS {$this->links_table}");
    }
}
