<?php
/**
 * Keyword Generator Class
 * Automatically generates keywords from post/page titles
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Keyword_Generator {

    /**
     * Generate keywords from title
     */
    public function generate_from_title($title) {
        if (empty($title)) {
            return '';
        }

        $keywords = array();

        // Add the full title
        $keywords[] = $title;

        // Remove common words and split into phrases
        $title_lower = strtolower($title);

        // Common stop words to remove
        $stop_words = array(
            'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for',
            'of', 'with', 'by', 'from', 'up', 'about', 'into', 'through', 'during',
            'is', 'are', 'was', 'were', 'be', 'been', 'being', 'have', 'has', 'had',
            'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might',
            'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it', 'we', 'they'
        );

        // Get words from title
        $words = preg_split('/[\s\-_]+/', $title_lower);
        $words = array_filter($words, function($word) use ($stop_words) {
            return strlen($word) > 2 && !in_array($word, $stop_words) && !is_numeric($word);
        });

        // Add individual important words
        foreach ($words as $word) {
            $keywords[] = $word;
        }

        // Generate 2-word phrases
        $words_array = array_values($words);
        for ($i = 0; $i < count($words_array) - 1; $i++) {
            $phrase = $words_array[$i] . ' ' . $words_array[$i + 1];
            $keywords[] = $phrase;
        }

        // Generate 3-word phrases for longer titles
        if (count($words_array) >= 3) {
            for ($i = 0; $i < count($words_array) - 2; $i++) {
                $phrase = $words_array[$i] . ' ' . $words_array[$i + 1] . ' ' . $words_array[$i + 2];
                $keywords[] = $phrase;
            }
        }

        // Remove duplicates and empty values
        $keywords = array_unique(array_filter($keywords));

        // Limit to top 15 most relevant keywords (prioritize longer phrases)
        usort($keywords, function($a, $b) {
            $word_count_a = str_word_count($a);
            $word_count_b = str_word_count($b);

            if ($word_count_a === $word_count_b) {
                return strlen($b) - strlen($a);
            }

            return $word_count_b - $word_count_a;
        });

        $keywords = array_slice($keywords, 0, 15);

        return implode(', ', $keywords);
    }

    /**
     * Generate keywords from URL slug
     */
    public function generate_from_url($url) {
        $path = parse_url($url, PHP_URL_PATH);
        $slug = basename($path);

        // Remove file extensions
        $slug = preg_replace('/\.(html|php|htm)$/', '', $slug);

        // Replace hyphens and underscores with spaces
        $text = str_replace(array('-', '_'), ' ', $slug);

        // Generate keywords from the cleaned text
        return $this->generate_from_title($text);
    }

    /**
     * Merge auto-generated and custom keywords
     */
    public function merge_keywords($auto_keywords, $custom_keywords) {
        $all_keywords = array();

        // Parse custom keywords
        if (!empty($custom_keywords)) {
            $custom = array_map('trim', explode(',', $custom_keywords));
            $all_keywords = array_merge($all_keywords, $custom);
        }

        // Parse auto keywords
        if (!empty($auto_keywords)) {
            $auto = array_map('trim', explode(',', $auto_keywords));
            $all_keywords = array_merge($all_keywords, $auto);
        }

        // Remove duplicates and empty values
        $all_keywords = array_unique(array_filter($all_keywords));

        return $all_keywords;
    }

    /**
     * Get combined keywords string
     */
    public function get_combined_keywords($auto_keywords, $custom_keywords) {
        $merged = $this->merge_keywords($auto_keywords, $custom_keywords);
        return implode(', ', $merged);
    }
}
