<?php
/**
 * Link Suggester Class
 * Main class for generating internal link suggestions using Supabase
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Link_Suggester {

    private $matcher;
    private $database;
    private $keyword_generator;

    public function __construct() {
        $this->matcher = new SEOINUX_Keyword_Matcher();
        $this->database = new SEOINUX_Database();
        $this->keyword_generator = new SEOINUX_Keyword_Generator();
    }

    /**
     * Get link suggestions for content
     *
     * @param string $content Content to analyze
     * @param int $exclude_post_id Post ID to exclude from suggestions
     * @return array Array of suggestions
     */
    public function get_suggestions($content, $exclude_post_id = 0) {
        // Get all links from Supabase
        $links = $this->get_all_links();

        if (empty($links)) {
            return array();
        }

        // Prepare links with combined keywords (auto + custom)
        $prepared_links = array();
        foreach ($links as $link) {
            $combined_keywords = $this->keyword_generator->get_combined_keywords(
                $link['auto_keywords'] ?? '',
                $link['keywords'] ?? ''
            );

            $prepared_links[] = array(
                'url' => $link['url'],
                'title' => $link['title'],
                'keywords' => $combined_keywords
            );
        }

        // Find matches
        $matches = $this->matcher->find_matches($content, $prepared_links, $exclude_post_id);

        // Limit results
        $max_suggestions = get_option('seoinux_max_suggestions', 5);
        $matches = array_slice($matches, 0, $max_suggestions);

        return $matches;
    }

    /**
     * Get all links from database
     *
     * @return array Array of links with keywords
     */
    private function get_all_links() {
        return $this->database->get_all_links();
    }

    /**
     * Get suggestion for specific text
     *
     * @param string $text Text to find link for
     * @return array|null Suggestion or null
     */
    public function get_suggestion_for_text($text) {
        $links = $this->get_all_links();

        if (empty($links)) {
            return null;
        }

        // Prepare links with combined keywords
        $prepared_links = array();
        foreach ($links as $link) {
            $combined_keywords = $this->keyword_generator->get_combined_keywords(
                $link['auto_keywords'] ?? '',
                $link['keywords'] ?? ''
            );

            $prepared_links[] = array(
                'url' => $link['url'],
                'title' => $link['title'],
                'keywords' => $combined_keywords
            );
        }

        return $this->matcher->find_best_match_for_text($text, $prepared_links);
    }

    /**
     * Get statistics about suggestions
     *
     * @return array Statistics data
     */
    public function get_statistics() {
        $links = $this->database->get_all_links();
        $total_links = count($links);
        $with_custom_keywords = 0;

        foreach ($links as $link) {
            if (!empty($link['keywords'])) {
                $with_custom_keywords++;
            }
        }

        return array(
            'total_links' => $total_links,
            'with_custom_keywords' => $with_custom_keywords,
            'with_auto_keywords' => $total_links
        );
    }
}
