<?php
/**
 * Supabase Connector Class
 * Handles all Supabase database operations
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Supabase_Connector {

    private $supabase_url;
    private $supabase_key;

    public function __construct() {
        $this->supabase_url = defined('SEOINUX_SUPABASE_URL') ? SEOINUX_SUPABASE_URL : '';
        $this->supabase_key = defined('SEOINUX_SUPABASE_KEY') ? SEOINUX_SUPABASE_KEY : '';
    }

    /**
     * Make a request to Supabase REST API
     */
    private function make_request($endpoint, $method = 'GET', $data = null) {
        if (empty($this->supabase_url) || empty($this->supabase_key)) {
            error_log('SEOINUX Supabase: Credentials not configured');
            return array('error' => 'Supabase credentials not configured');
        }

        $url = trailingslashit($this->supabase_url) . 'rest/v1/' . $endpoint;

        $headers = array(
            'apikey' => $this->supabase_key,
            'Authorization' => 'Bearer ' . $this->supabase_key,
            'Content-Type' => 'application/json',
            'Prefer' => 'return=representation'
        );

        $args = array(
            'method' => $method,
            'headers' => $headers,
            'timeout' => 30
        );

        if ($data !== null) {
            $args['body'] = json_encode($data);
        }

        error_log('SEOINUX Supabase: ' . $method . ' ' . $endpoint);

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            error_log('SEOINUX Supabase Error: ' . $response->get_error_message());
            return array('error' => $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $status = wp_remote_retrieve_response_code($response);

        error_log('SEOINUX Supabase: Response status ' . $status);

        $result = json_decode($body, true);

        if ($status >= 400) {
            $error_msg = isset($result['message']) ? $result['message'] : 'Request failed';
            error_log('SEOINUX Supabase Error: ' . $error_msg . ' - Body: ' . $body);
            return array('error' => $error_msg);
        }

        return $result;
    }

    /**
     * Insert or update a link
     */
    public function upsert_link($url, $title, $auto_keywords = '', $keywords = null) {
        // Check if link exists
        $existing = $this->get_link_by_url($url);

        if ($existing && !isset($existing['error'])) {
            // Update existing link
            $update_data = array(
                'title' => $title
            );

            // Only update auto_keywords if provided
            if (!empty($auto_keywords)) {
                $update_data['auto_keywords'] = $auto_keywords;
            }

            // Only update keywords if explicitly provided (not null)
            if ($keywords !== null) {
                $update_data['keywords'] = $keywords;
            }

            return $this->make_request(
                'seoinux_links?url=eq.' . urlencode($url),
                'PATCH',
                $update_data
            );
        } else {
            // Insert new link
            $insert_data = array(
                'url' => $url,
                'title' => $title,
                'auto_keywords' => $auto_keywords,
                'keywords' => $keywords !== null ? $keywords : ''
            );

            return $this->make_request(
                'seoinux_links',
                'POST',
                $insert_data
            );
        }
    }

    /**
     * Get link by URL
     */
    public function get_link_by_url($url) {
        $result = $this->make_request('seoinux_links?url=eq.' . urlencode($url));

        if (isset($result['error'])) {
            return $result;
        }

        return !empty($result) ? $result[0] : null;
    }

    /**
     * Get all links
     */
    public function get_all_links($limit = null, $offset = 0) {
        $endpoint = 'seoinux_links?order=title.asc';

        if ($limit !== null) {
            $endpoint .= '&limit=' . intval($limit) . '&offset=' . intval($offset);
        }

        return $this->make_request($endpoint);
    }

    /**
     * Get links with keywords
     */
    public function get_links_with_keywords() {
        $result = $this->make_request('seoinux_links?or=(keywords.neq.,auto_keywords.neq.)');

        if (isset($result['error'])) {
            return array();
        }

        return $result;
    }

    /**
     * Search links
     */
    public function search_links($search_term, $limit = 20, $offset = 0) {
        $search_term = sanitize_text_field($search_term);
        $encoded = urlencode($search_term);

        $endpoint = 'seoinux_links?or=(url.ilike.*' . $encoded . '*,title.ilike.*' . $encoded . '*,keywords.ilike.*' . $encoded . '*)';
        $endpoint .= '&order=title.asc&limit=' . intval($limit) . '&offset=' . intval($offset);

        return $this->make_request($endpoint);
    }

    /**
     * Count all links
     */
    public function count_links($search_term = '') {
        $endpoint = 'seoinux_links?select=count';

        if (!empty($search_term)) {
            $encoded = urlencode($search_term);
            $endpoint .= '&or=(url.ilike.*' . $encoded . '*,title.ilike.*' . $encoded . '*,keywords.ilike.*' . $encoded . '*)';
        }

        $result = $this->make_request($endpoint);

        if (isset($result['error']) || empty($result)) {
            return 0;
        }

        return isset($result[0]['count']) ? intval($result[0]['count']) : 0;
    }

    /**
     * Update keywords for a URL
     */
    public function update_keywords($url, $keywords) {
        return $this->make_request(
            'seoinux_links?url=eq.' . urlencode($url),
            'PATCH',
            array('keywords' => $keywords)
        );
    }

    /**
     * Delete a link
     */
    public function delete_link($url) {
        return $this->make_request(
            'seoinux_links?url=eq.' . urlencode($url),
            'DELETE'
        );
    }

    /**
     * Get statistics
     */
    public function get_statistics() {
        $all_links = $this->get_all_links();

        if (isset($all_links['error'])) {
            return array(
                'total_links' => 0,
                'links_with_keywords' => 0,
                'links_without_keywords' => 0,
                'completion_percentage' => 0
            );
        }

        $total = count($all_links);
        $with_keywords = 0;

        foreach ($all_links as $link) {
            if (!empty($link['keywords']) || !empty($link['auto_keywords'])) {
                $with_keywords++;
            }
        }

        $without_keywords = $total - $with_keywords;
        $completion = $total > 0 ? round(($with_keywords / $total) * 100, 1) : 0;

        return array(
            'total_links' => $total,
            'links_with_keywords' => $with_keywords,
            'links_without_keywords' => $without_keywords,
            'completion_percentage' => $completion
        );
    }

    /**
     * Clear all links (for testing/reset)
     */
    public function clear_all_links() {
        return $this->make_request('seoinux_links', 'DELETE');
    }
}
