<?php
/**
 * Plugin Name: SEOINUX SEO Internal Link Creator
 * Plugin URI: https://seoinux.com
 * Description: Automatically suggests internal links from sitemap.xml with auto-generated keywords from titles. Uses WordPress database for keyword management.
 * Version: 2.0.0
 * Author: SEOINUX
 * Author URI: https://seoinux.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: seoinux-seo-internal-link
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SEOINUX_VERSION', '2.0.0');
define('SEOINUX_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SEOINUX_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include required files
require_once SEOINUX_PLUGIN_DIR . 'includes/class-database.php';
require_once SEOINUX_PLUGIN_DIR . 'includes/class-keyword-generator.php';
require_once SEOINUX_PLUGIN_DIR . 'includes/class-sitemap-parser.php';
require_once SEOINUX_PLUGIN_DIR . 'includes/class-keyword-matcher.php';
require_once SEOINUX_PLUGIN_DIR . 'includes/class-link-suggester.php';
require_once SEOINUX_PLUGIN_DIR . 'admin/class-admin-interface.php';

/**
 * Main Plugin Class
 */
class SEOINUX_Internal_Link_Creator {

    private static $instance = null;
    private $database;
    private $keyword_generator;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->database = new SEOINUX_Database();
        $this->keyword_generator = new SEOINUX_Keyword_Generator();
        $this->init_hooks();
    }

    private function init_hooks() {
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Initialize admin interface
        if (is_admin()) {
            new SEOINUX_Admin_Interface();
        }

        // Enqueue scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_assets'));

        // AJAX handlers
        add_action('wp_ajax_seoinux_get_suggestions', array($this, 'ajax_get_suggestions'));
        add_action('wp_ajax_seoinux_get_suggestions_for_text', array($this, 'ajax_get_suggestions_for_text'));
        add_action('wp_ajax_seoinux_get_all_links', array($this, 'ajax_get_all_links'));
        add_action('wp_ajax_seoinux_refresh_sitemap', array($this, 'ajax_refresh_sitemap'));
        add_action('wp_ajax_seoinux_save_keywords', array($this, 'ajax_save_keywords'));
        add_action('wp_ajax_seoinux_regenerate_keywords', array($this, 'ajax_regenerate_keywords'));

        // Add meta box for classic editor
        add_action('add_meta_boxes', array($this, 'add_meta_box'));

        // Schedule sitemap refresh
        if (!wp_next_scheduled('seoinux_refresh_sitemap_cron')) {
            wp_schedule_event(time(), 'daily', 'seoinux_refresh_sitemap_cron');
        }
        add_action('seoinux_refresh_sitemap_cron', array($this, 'refresh_sitemap_data'));
    }

    public function activate() {
        // Create database tables
        $this->database->create_tables();

        // Set default options
        if (!get_option('seoinux_sitemap_url')) {
            update_option('seoinux_sitemap_url', site_url('/sitemap.xml'));
        }
        if (!get_option('seoinux_max_suggestions')) {
            update_option('seoinux_max_suggestions', 5);
        }
        if (!get_option('seoinux_auto_refresh')) {
            update_option('seoinux_auto_refresh', 1);
        }
        if (!get_option('seoinux_auto_generate_keywords')) {
            update_option('seoinux_auto_generate_keywords', 1);
        }
    }

    public function deactivate() {
        // Clear scheduled events
        wp_clear_scheduled_hook('seoinux_refresh_sitemap_cron');
    }

    public function enqueue_admin_assets($hook) {
        // Only load on plugin pages and post editor
        if ($hook !== 'toplevel_page_seoinux-link-creator' &&
            $hook !== 'link-creator_page_seoinux-manage-keywords' &&
            $hook !== 'link-creator_page_seoinux-settings' &&
            $hook !== 'post.php' &&
            $hook !== 'post-new.php') {
            return;
        }

        wp_enqueue_style(
            'seoinux-admin-style',
            SEOINUX_PLUGIN_URL . 'assets/css/admin-style.css',
            array(),
            SEOINUX_VERSION
        );

        wp_enqueue_script(
            'seoinux-admin-script',
            SEOINUX_PLUGIN_URL . 'assets/js/admin-script.js',
            array('jquery'),
            SEOINUX_VERSION,
            true
        );

        wp_localize_script('seoinux-admin-script', 'seoinuxAjax', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seoinux_nonce')
        ));

        // Load editor script on post edit pages (for both Classic and Gutenberg)
        if ($hook === 'post.php' || $hook === 'post-new.php') {
            wp_enqueue_script(
                'seoinux-editor-script',
                SEOINUX_PLUGIN_URL . 'assets/js/editor-script.js',
                array('jquery', 'wp-element', 'wp-data'),
                SEOINUX_VERSION,
                true
            );

            wp_localize_script('seoinux-editor-script', 'seoinuxEditor', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('seoinux_nonce')
            ));
        }
    }

    public function enqueue_editor_assets() {
        wp_enqueue_script(
            'seoinux-editor-script',
            SEOINUX_PLUGIN_URL . 'assets/js/editor-script.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-data'),
            SEOINUX_VERSION,
            true
        );

        wp_localize_script('seoinux-editor-script', 'seoinuxEditor', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seoinux_nonce')
        ));
    }

    public function add_meta_box() {
        add_meta_box(
            'seoinux_suggestions_meta_box',
            'SEOINUX Link Suggestions',
            array($this, 'render_meta_box'),
            array('post', 'page'),
            'side',
            'high'
        );
    }

    public function render_meta_box($post) {
        wp_nonce_field('seoinux_meta_box', 'seoinux_meta_box_nonce');
        ?>
        <div id="seoinux-suggestions-container" style="margin: -12px; margin-top: -6px;">
            <!-- Header -->
            <div style="background: linear-gradient(135deg, #0073aa 0%, #005177 100%); padding: 15px; color: white;">
                <strong style="display: block; font-size: 14px; margin-bottom: 5px;">SEOINUX Link Creator</strong>
                <small style="opacity: 0.9;">Select text, then click a link to insert</small>
            </div>

            <!-- Search Box -->
            <div style="padding: 15px; border-bottom: 1px solid #ddd; background: #f9f9f9;">
                <input type="text" id="seoinux-search-links" placeholder="🔍 Search links..."
                    style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px;">
            </div>

            <!-- Selected Text Info -->
            <div id="seoinux-selected-text-info" style="display: none; padding: 10px 15px; background: #e7f5ff; border-bottom: 1px solid #b8daff;">
                <small style="color: #004085;">
                    <strong>Selected:</strong> <span id="seoinux-selected-text-display"></span>
                </small>
                <button type="button" id="seoinux-clear-selection" style="float: right; padding: 2px 8px; font-size: 11px;" class="button button-small">Clear</button>
            </div>

            <!-- Loading State -->
            <div id="seoinux-loading" style="padding: 40px 15px; text-align: center; color: #666;">
                <span class="spinner is-active" style="float: none; margin: 0 auto 10px;"></span>
                <p style="margin: 0; font-size: 13px;">Loading links...</p>
            </div>

            <!-- Links List -->
            <div id="seoinux-links-list" style="display: none; max-height: 500px; overflow-y: auto;">
                <!-- Links will be populated here -->
            </div>

            <!-- Empty State -->
            <div id="seoinux-empty-state" style="display: none; padding: 40px 15px; text-align: center; color: #666;">
                <span class="dashicons dashicons-admin-links" style="font-size: 48px; opacity: 0.3; width: 48px; height: 48px;"></span>
                <p style="margin: 10px 0 0; font-size: 13px;">No links found. Click "Refresh Sitemap" on the main page.</p>
            </div>

            <!-- How to Use -->
            <div style="padding: 15px; background: #f9f9f9; border-top: 1px solid #ddd; font-size: 12px; color: #666;">
                <strong style="display: block; margin-bottom: 8px; color: #333;">How to use:</strong>
                1. Highlight text in your content<br>
                2. Click any link below to insert it<br>
                3. Use search to filter links
            </div>
        </div>
        <?php
    }

    public function ajax_get_suggestions() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        $content = isset($_POST['content']) ? wp_kses_post($_POST['content']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        $suggester = new SEOINUX_Link_Suggester();
        $suggestions = $suggester->get_suggestions($content, $post_id);

        wp_send_json_success($suggestions);
    }

    public function ajax_get_suggestions_for_text() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        $text = isset($_POST['text']) ? sanitize_text_field($_POST['text']) : '';

        if (empty($text)) {
            wp_send_json_error(array('message' => 'No text provided'));
        }

        $suggester = new SEOINUX_Link_Suggester();
        $all_links = $this->database->get_all_links();

        if (empty($all_links)) {
            wp_send_json_success(array());
        }

        $prepared_links = array();
        foreach ($all_links as $link) {
            $combined_keywords = $this->keyword_generator->get_combined_keywords(
                $link['auto_keywords'] ?? '',
                $link['keywords'] ?? ''
            );

            $prepared_links[] = array(
                'url' => $link['url'],
                'title' => $link['title'],
                'keywords' => $combined_keywords
            );
        }

        $matcher = new SEOINUX_Keyword_Matcher();
        $best_matches = array();

        foreach ($prepared_links as $link) {
            $match = $this->check_text_match($text, $link);
            if ($match) {
                $best_matches[] = $match;
            }
        }

        usort($best_matches, function($a, $b) {
            return $b['score'] - $a['score'];
        });

        $best_matches = array_slice($best_matches, 0, 5);

        wp_send_json_success($best_matches);
    }

    private function check_text_match($text, $link) {
        $text_lower = strtolower(trim($text));
        $keywords = array_map('trim', explode(',', $link['keywords']));

        foreach ($keywords as $keyword) {
            $keyword_lower = strtolower($keyword);

            if ($keyword_lower === $text_lower) {
                return array(
                    'keyword' => $keyword,
                    'url' => $link['url'],
                    'title' => $link['title'],
                    'score' => 100
                );
            }

            if (strpos($text_lower, $keyword_lower) !== false) {
                return array(
                    'keyword' => $keyword,
                    'url' => $link['url'],
                    'title' => $link['title'],
                    'score' => 80
                );
            }

            if (strpos($keyword_lower, $text_lower) !== false) {
                return array(
                    'keyword' => $keyword,
                    'url' => $link['url'],
                    'title' => $link['title'],
                    'score' => 60
                );
            }

            $similarity = 0;
            similar_text($keyword_lower, $text_lower, $similarity);
            if ($similarity > 70) {
                return array(
                    'keyword' => $keyword,
                    'url' => $link['url'],
                    'title' => $link['title'],
                    'score' => $similarity
                );
            }
        }

        return null;
    }

    public function ajax_get_all_links() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        $all_links = $this->database->get_all_links();

        if (empty($all_links)) {
            wp_send_json_success(array());
        }

        $prepared_links = array();
        foreach ($all_links as $link) {
            $combined_keywords = $this->keyword_generator->get_combined_keywords(
                $link['auto_keywords'] ?? '',
                $link['keywords'] ?? ''
            );

            $prepared_links[] = array(
                'url' => $link['url'],
                'title' => $link['title'],
                'keywords' => $combined_keywords
            );
        }

        wp_send_json_success($prepared_links);
    }

    public function ajax_refresh_sitemap() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        $result = $this->refresh_sitemap_data();

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    public function ajax_save_keywords() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';
        $keywords = isset($_POST['keywords']) ? sanitize_textarea_field($_POST['keywords']) : '';

        $result = $this->database->update_keywords($url, $keywords);

        if (!isset($result['error'])) {
            wp_send_json_success(array('message' => 'Keywords saved successfully'));
        } else {
            wp_send_json_error(array('message' => 'Failed to save keywords: ' . $result['error']));
        }
    }

    public function ajax_regenerate_keywords() {
        check_ajax_referer('seoinux_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';

        // Get the link
        $link = $this->database->get_link_by_url($url);

        if (!$link || isset($link['error'])) {
            wp_send_json_error(array('message' => 'Link not found'));
            return;
        }

        // Regenerate keywords from title
        $auto_keywords = $this->keyword_generator->generate_from_title($link['title']);

        // Update the link
        $result = $this->database->upsert_link($url, $link['title'], $auto_keywords, $link['keywords']);

        if (!isset($result['error'])) {
            wp_send_json_success(array(
                'message' => 'Keywords regenerated',
                'keywords' => $auto_keywords
            ));
        } else {
            wp_send_json_error(array('message' => 'Failed to regenerate keywords'));
        }
    }

    public function refresh_sitemap_data() {
        $parser = new SEOINUX_Sitemap_Parser();
        $sitemap_url = get_option('seoinux_sitemap_url', site_url('/sitemap.xml'));

        error_log('SEOINUX: Attempting to parse sitemap: ' . $sitemap_url);

        $links = $parser->parse($sitemap_url);

        error_log('SEOINUX: Parser returned ' . count($links) . ' links');

        if (empty($links)) {
            error_log('SEOINUX: No links found in sitemap');
            return array(
                'success' => false,
                'message' => 'Failed to parse sitemap or sitemap is empty. Check error logs for details.'
            );
        }

        $auto_generate = get_option('seoinux_auto_generate_keywords', 1);
        $inserted = 0;
        $errors = 0;

        error_log('SEOINUX: Starting to process ' . count($links) . ' links');

        foreach ($links as $link) {
            // Generate auto keywords if enabled
            $auto_keywords = '';
            if ($auto_generate) {
                $auto_keywords = $this->keyword_generator->generate_from_title($link['title']);
                error_log('SEOINUX: Generated keywords for "' . $link['title'] . '": ' . substr($auto_keywords, 0, 50) . '...');
            }

            // Upsert link (preserves existing custom keywords)
            $result = $this->database->upsert_link($link['url'], $link['title'], $auto_keywords);

            if (!isset($result['error'])) {
                $inserted++;
                error_log('SEOINUX: Successfully saved: ' . $link['url']);
            } else {
                $errors++;
                error_log('SEOINUX: Failed to save ' . $link['url'] . ': ' . $result['error']);
            }
        }

        error_log('SEOINUX: Completed - Inserted: ' . $inserted . ', Errors: ' . $errors);

        if ($errors > 0 && $inserted === 0) {
            return array(
                'success' => false,
                'message' => 'Failed to save links to database. Check error logs.',
                'inserted' => $inserted,
                'errors' => $errors
            );
        }

        return array(
            'success' => true,
            'message' => "Sitemap refreshed: $inserted links saved" . ($errors > 0 ? ", $errors errors" : ''),
            'inserted' => $inserted,
            'errors' => $errors
        );
    }

    public function get_database() {
        return $this->database;
    }

    public function get_keyword_generator() {
        return $this->keyword_generator;
    }
}

// Initialize the plugin
function seoinux_init() {
    return SEOINUX_Internal_Link_Creator::get_instance();
}

// Start the plugin
seoinux_init();
