<?php
/**
 * Cache Handler Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Cache_Handler {

    private $cache_dir;

    public function __construct() {
        $this->cache_dir = WP_CONTENT_DIR . '/cache/seoinux-cache-pro';
    }

    public function clear_all() {
        $this->clear_page_cache();
        $this->clear_object_cache();

        // Clear WordPress transients
        delete_expired_transients(true);

        // Log activity
        $this->log_activity('Cache cleared successfully');

        return true;
    }

    private function log_activity($message) {
        $activities = get_option('seoinux_activities', array());

        // Add new activity
        array_unshift($activities, array(
            'action' => $message,
            'time' => current_time('timestamp'),
            'status' => 'success'
        ));

        // Keep only last 10 activities
        $activities = array_slice($activities, 0, 10);

        update_option('seoinux_activities', $activities);
    }

    private function clear_page_cache() {
        if (!file_exists($this->cache_dir)) {
            return;
        }

        $files = glob($this->cache_dir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
    }

    private function clear_object_cache() {
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
    }

    public function get_hit_rate() {
        $hits = intval(get_option('seoinux_cache_hits', 0));
        $misses = intval(get_option('seoinux_cache_misses', 0));
        $total = $hits + $misses;

        if ($total == 0) {
            return '0%';
        }

        $rate = ($hits / $total) * 100;
        return number_format($rate, 1) . '%';
    }

    public function get_avg_load_time() {
        $load_times = get_option('seoinux_load_times', array());

        if (empty($load_times) || !is_array($load_times)) {
            return '0.00s';
        }

        $avg = array_sum($load_times) / count($load_times);
        return number_format($avg, 2) . 's';
    }

    public function get_bandwidth_saved() {
        $saved = floatval(get_option('seoinux_bandwidth_saved', 0));

        if ($saved < 1024) {
            return number_format($saved, 2) . ' KB';
        } elseif ($saved < 1024 * 1024) {
            return number_format($saved / 1024, 2) . ' MB';
        } else {
            return number_format($saved / (1024 * 1024), 2) . ' GB';
        }
    }

    public function get_cache($key) {
        $file = $this->cache_dir . '/' . md5($key) . '.cache';

        if (!file_exists($file)) {
            $this->record_miss();
            return false;
        }

        $data = unserialize(file_get_contents($file));

        if ($data['expires'] < time()) {
            unlink($file);
            $this->record_miss();
            return false;
        }

        $this->record_hit();
        return $data['content'];
    }

    private function record_hit() {
        $hits = get_option('seoinux_cache_hits', 0);
        update_option('seoinux_cache_hits', $hits + 1);
    }

    private function record_miss() {
        $misses = get_option('seoinux_cache_misses', 0);
        update_option('seoinux_cache_misses', $misses + 1);
    }

    public function record_page_load_time($load_time) {
        $load_times = get_option('seoinux_load_times', array());
        $load_times[] = $load_time;

        // Keep only last 100 load times
        if (count($load_times) > 100) {
            $load_times = array_slice($load_times, -100);
        }

        update_option('seoinux_load_times', $load_times);
    }

    public function set_cache($key, $content, $duration = 3600) {
        if (!file_exists($this->cache_dir)) {
            wp_mkdir_p($this->cache_dir);
        }

        $file = $this->cache_dir . '/' . md5($key) . '.cache';
        $data = array(
            'content' => $content,
            'expires' => time() + $duration
        );

        file_put_contents($file, serialize($data));

        // Log activity
        $this->log_activity('Cache entry created');

        return true;
    }

}
