<?php
/**
 * Settings Handler Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEOINUX_Settings {

    private $option_prefix = 'seoinux_';

    public function get_all() {
        return array(
            'page_cache' => get_option($this->option_prefix . 'page_cache', true),
            'browser_cache' => get_option($this->option_prefix . 'browser_cache', true),
            'object_cache' => get_option($this->option_prefix . 'object_cache', false),
            'gzip_compression' => get_option($this->option_prefix . 'gzip_compression', true),
            'cache_duration' => get_option($this->option_prefix . 'cache_duration', 604800),
            'lazy_load' => get_option($this->option_prefix . 'lazy_load', true),
            'webp_conversion' => get_option($this->option_prefix . 'webp_conversion', true),
            'auto_resize' => get_option($this->option_prefix . 'auto_resize', true),
            'compression_level' => get_option($this->option_prefix . 'compression_level', 'balanced'),
            'max_width' => get_option($this->option_prefix . 'max_width', '1920'),
            'minify_html' => get_option($this->option_prefix . 'minify_html', true),
            'minify_css' => get_option($this->option_prefix . 'minify_css', true),
            'minify_js' => get_option($this->option_prefix . 'minify_js', true),
            'combine_css' => get_option($this->option_prefix . 'combine_css', true),
            'combine_js' => get_option($this->option_prefix . 'combine_js', false),
            'defer_js' => get_option($this->option_prefix . 'defer_js', true),
            'lazy_load_videos' => get_option($this->option_prefix . 'lazy_load_videos', true),
            'preload_fonts' => get_option($this->option_prefix . 'preload_fonts', true),
            'cdn_enabled' => get_option($this->option_prefix . 'cdn_enabled', false),
            'cdn_url' => get_option($this->option_prefix . 'cdn_url', ''),
            'database_optimization' => get_option($this->option_prefix . 'database_optimization', true),
            'remove_query_strings' => get_option($this->option_prefix . 'remove_query_strings', true),
            'dns_prefetch' => get_option($this->option_prefix . 'dns_prefetch', true),
        );
    }

    public function save($settings) {
        foreach ($settings as $key => $value) {
            update_option($this->option_prefix . $key, $value);
        }

        // Apply settings
        $this->apply_settings();

        return true;
    }

    private function apply_settings() {
        // Clear cache when settings change
        $cache_handler = new SEOINUX_Cache_Handler();
        $cache_handler->clear_all();

        // Update .htaccess for browser caching if enabled
        if (get_option($this->option_prefix . 'browser_cache')) {
            $this->update_htaccess();
        }

        // Log activity
        $this->log_activity('Settings updated successfully');
    }

    private function log_activity($message) {
        $activities = get_option('seoinux_activities', array());

        // Add new activity
        array_unshift($activities, array(
            'action' => $message,
            'time' => current_time('timestamp'),
            'status' => 'success'
        ));

        // Keep only last 10 activities
        $activities = array_slice($activities, 0, 10);

        update_option('seoinux_activities', $activities);
    }

    private function update_htaccess() {
        $htaccess_file = ABSPATH . '.htaccess';

        if (!is_writable($htaccess_file)) {
            return;
        }

        $htaccess_content = file_get_contents($htaccess_file);

        // Remove existing SEOINUX rules
        $htaccess_content = preg_replace('/#\s*BEGIN SEOINUX Cache.*?#\s*END SEOINUX Cache/s', '', $htaccess_content);

        // Add new rules
        $rules = "# BEGIN SEOINUX Cache\n";
        $rules .= "<IfModule mod_expires.c>\n";
        $rules .= "ExpiresActive On\n";
        $rules .= "ExpiresByType image/jpg \"access plus 1 year\"\n";
        $rules .= "ExpiresByType image/jpeg \"access plus 1 year\"\n";
        $rules .= "ExpiresByType image/gif \"access plus 1 year\"\n";
        $rules .= "ExpiresByType image/png \"access plus 1 year\"\n";
        $rules .= "ExpiresByType image/webp \"access plus 1 year\"\n";
        $rules .= "ExpiresByType text/css \"access plus 1 month\"\n";
        $rules .= "ExpiresByType application/javascript \"access plus 1 month\"\n";
        $rules .= "ExpiresByType application/pdf \"access plus 1 month\"\n";
        $rules .= "ExpiresByType text/html \"access plus 1 hour\"\n";
        $rules .= "</IfModule>\n";

        if (get_option($this->option_prefix . 'gzip_compression')) {
            $rules .= "<IfModule mod_deflate.c>\n";
            $rules .= "AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/x-javascript\n";
            $rules .= "</IfModule>\n";
        }

        $rules .= "# END SEOINUX Cache\n\n";

        $htaccess_content = $rules . $htaccess_content;
        file_put_contents($htaccess_file, $htaccess_content);
    }

    public function get($key) {
        return get_option($this->option_prefix . $key);
    }

    public function set($key, $value) {
        return update_option($this->option_prefix . $key, $value);
    }
}
