<?php
/**
 * Plugin Name: SEOINUX Cache Pro
 * Plugin URI: https://seoinux.com/cache-pro
 * Description: WordPress Performance & Optimisation Plugin with advanced caching, image optimisation, and PageSpeed tools
 * Version: 1.8
 * Author: SEOINUX
 * Author URI: https://seoinux.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: seoinux-cache-pro
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Track page load start time
if (!defined('WP_START_TIMESTAMP')) {
    define('WP_START_TIMESTAMP', microtime(true));
}

// Define plugin constants
define('SEOINUX_CACHE_VERSION', '1.8');
define('SEOINUX_CACHE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SEOINUX_CACHE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SEOINUX_CACHE_PLUGIN_FILE', __FILE__);

class SEOINUX_Cache_Pro {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->init_hooks();
        $this->load_dependencies();
    }

    private function init_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_seoinux_clear_cache', array($this, 'clear_cache'));
        add_action('wp_ajax_seoinux_optimize_images', array($this, 'optimize_images'));
        add_action('wp_ajax_seoinux_save_settings', array($this, 'save_settings'));
        add_action('wp_ajax_seoinux_get_stats', array($this, 'get_stats_ajax'));

        // Register activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Track page load times and implement caching
        add_action('template_redirect', array($this, 'maybe_serve_cache'), 1);
        add_action('shutdown', array($this, 'track_page_load'));
    }

    public function maybe_serve_cache() {
        if (is_admin() || defined('DOING_AJAX') || defined('DOING_CRON') || is_user_logged_in()) {
            return;
        }

        if (!get_option('seoinux_page_cache', true)) {
            return;
        }

        $cache_handler = new SEOINUX_Cache_Handler();
        $cache_key = $_SERVER['REQUEST_URI'];

        $cached_content = $cache_handler->get_cache($cache_key);

        if ($cached_content !== false) {
            echo $cached_content;
            exit;
        }

        // Start output buffering to cache the page
        ob_start(function($buffer) use ($cache_handler, $cache_key) {
            if (strlen($buffer) > 0 && !is_404() && !is_search()) {
                $duration = intval(get_option('seoinux_cache_duration', 604800));
                $cache_handler->set_cache($cache_key, $buffer, $duration);
            }
            return $buffer;
        });
    }

    public function track_page_load() {
        if (is_admin() || defined('DOING_AJAX') || defined('DOING_CRON')) {
            return;
        }

        // Calculate page generation time
        if (defined('WP_START_TIMESTAMP')) {
            $load_time = microtime(true) - WP_START_TIMESTAMP;
            $cache_handler = new SEOINUX_Cache_Handler();
            $cache_handler->record_page_load_time($load_time);
        }
    }

    private function load_dependencies() {
        require_once SEOINUX_CACHE_PLUGIN_DIR . 'includes/class-cache-handler.php';
        require_once SEOINUX_CACHE_PLUGIN_DIR . 'includes/class-image-optimizer.php';
        require_once SEOINUX_CACHE_PLUGIN_DIR . 'includes/class-settings.php';
    }

    public function add_admin_menu() {
        add_menu_page(
            __('SEOINUX Cache Pro', 'seoinux-cache-pro'),
            __('SEOINUX Cache', 'seoinux-cache-pro'),
            'manage_options',
            'seoinux-cache-pro',
            array($this, 'render_admin_page'),
            'dashicons-performance',
            30
        );
    }

    public function render_admin_page() {
        ?>
        <div class="wrap">
            <div id="seoinux-cache-pro-root" style="margin-top: 20px;"></div>
        </div>
        <?php
    }

    public function enqueue_admin_assets($hook) {
        if ('toplevel_page_seoinux-cache-pro' !== $hook) {
            return;
        }

        // Check if dist files exist
        $js_file = SEOINUX_CACHE_PLUGIN_DIR . 'dist/assets/index.js';
        $css_file = SEOINUX_CACHE_PLUGIN_DIR . 'dist/assets/index.css';

        if (file_exists($js_file)) {
            // Production mode - load built assets
            wp_enqueue_script(
                'seoinux-cache-pro-js',
                SEOINUX_CACHE_PLUGIN_URL . 'dist/assets/index.js',
                array(),
                SEOINUX_CACHE_VERSION,
                true
            );
            wp_script_add_data('seoinux-cache-pro-js', 'type', 'module');
        } else {
            // Development mode - load from Vite dev server
            wp_enqueue_script(
                'seoinux-cache-pro-vite-client',
                'http://localhost:5173/@vite/client',
                array(),
                null,
                true
            );
            wp_script_add_data('seoinux-cache-pro-vite-client', 'type', 'module');

            wp_enqueue_script(
                'seoinux-cache-pro-js',
                'http://localhost:5173/src/main.tsx',
                array(),
                null,
                true
            );
            wp_script_add_data('seoinux-cache-pro-js', 'type', 'module');
        }

        if (file_exists($css_file)) {
            wp_enqueue_style(
                'seoinux-cache-pro-css',
                SEOINUX_CACHE_PLUGIN_URL . 'dist/assets/index.css',
                array(),
                SEOINUX_CACHE_VERSION
            );
        }

        // Localize script with WordPress data
        wp_localize_script('seoinux-cache-pro-js', 'seoinuxCachePro', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seoinux_cache_nonce'),
            'siteUrl' => get_site_url(),
            'pluginUrl' => SEOINUX_CACHE_PLUGIN_URL,
            'settings' => $this->get_settings(),
            'stats' => $this->get_stats(),
        ));
    }

    public function clear_cache() {
        check_ajax_referer('seoinux_cache_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $cache_handler = new SEOINUX_Cache_Handler();
        $result = $cache_handler->clear_all();

        wp_send_json_success(array(
            'message' => 'Cache cleared successfully',
            'stats' => $this->get_stats()
        ));
    }

    public function optimize_images() {
        check_ajax_referer('seoinux_cache_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        $image_optimizer = new SEOINUX_Image_Optimizer();
        $result = $image_optimizer->optimize_all();

        wp_send_json_success(array(
            'message' => 'Images optimised successfully',
            'count' => $result['count'],
            'saved' => $result['saved']
        ));
    }

    public function save_settings() {
        check_ajax_referer('seoinux_cache_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        // Get settings from POST data
        $settings = array();
        foreach ($_POST as $key => $value) {
            if ($key !== 'action' && $key !== 'nonce') {
                $settings[$key] = $value;
            }
        }

        // If settings is still empty, try JSON body
        if (empty($settings)) {
            $json = file_get_contents('php://input');
            if (!empty($json)) {
                $settings = json_decode($json, true);
            }
        }

        if (empty($settings)) {
            wp_send_json_error(array('message' => 'No settings provided'));
            return;
        }

        $settings_handler = new SEOINUX_Settings();
        $settings_handler->save($settings);

        wp_send_json_success(array('message' => 'Settings saved successfully'));
    }

    private function get_settings() {
        $settings_handler = new SEOINUX_Settings();
        return $settings_handler->get_all();
    }

    private function get_stats() {
        $cache_handler = new SEOINUX_Cache_Handler();
        $activities = get_option('seoinux_activities', array());

        // Format activities for display
        $formatted_activities = array();
        foreach ($activities as $activity) {
            $time_diff = current_time('timestamp') - $activity['time'];

            if ($time_diff < 60) {
                $time_ago = 'Just now';
            } elseif ($time_diff < 3600) {
                $minutes = floor($time_diff / 60);
                $time_ago = $minutes . ' minute' . ($minutes > 1 ? 's' : '') . ' ago';
            } elseif ($time_diff < 86400) {
                $hours = floor($time_diff / 3600);
                $time_ago = $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
            } else {
                $days = floor($time_diff / 86400);
                $time_ago = $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
            }

            $formatted_activities[] = array(
                'action' => $activity['action'],
                'time' => $time_ago,
                'status' => $activity['status']
            );
        }

        return array(
            'cache_hit_rate' => $cache_handler->get_hit_rate(),
            'page_load_time' => $cache_handler->get_avg_load_time(),
            'images_optimized' => get_option('seoinux_images_optimized', 0),
            'bandwidth_saved' => $cache_handler->get_bandwidth_saved(),
            'activities' => $formatted_activities,
        );
    }

    public function get_stats_ajax() {
        check_ajax_referer('seoinux_cache_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }

        wp_send_json_success($this->get_stats());
    }

    public function activate() {
        // Create cache directory
        $cache_dir = WP_CONTENT_DIR . '/cache/seoinux-cache-pro';
        if (!file_exists($cache_dir)) {
            wp_mkdir_p($cache_dir);
        }

        // Set default options
        $defaults = array(
            'page_cache' => true,
            'browser_cache' => true,
            'object_cache' => false,
            'gzip_compression' => true,
            'cache_duration' => 604800,
            'lazy_load' => true,
            'webp_conversion' => true,
            'minify_html' => true,
            'minify_css' => true,
            'minify_js' => true,
        );

        foreach ($defaults as $key => $value) {
            if (get_option('seoinux_' . $key) === false) {
                add_option('seoinux_' . $key, $value);
            }
        }

        // Initialize stats
        if (get_option('seoinux_cache_hits') === false) {
            add_option('seoinux_cache_hits', 0);
        }
        if (get_option('seoinux_cache_misses') === false) {
            add_option('seoinux_cache_misses', 0);
        }
        if (get_option('seoinux_images_optimized') === false) {
            add_option('seoinux_images_optimized', 0);
        }
        if (get_option('seoinux_bandwidth_saved') === false) {
            add_option('seoinux_bandwidth_saved', 0);
        }
        if (get_option('seoinux_load_times') === false) {
            add_option('seoinux_load_times', array());
        }

        // Add initial activity
        $activities = array(
            array(
                'action' => 'SEOINUX Cache Pro activated',
                'time' => current_time('timestamp'),
                'status' => 'success'
            )
        );
        add_option('seoinux_activities', $activities);

        flush_rewrite_rules();
    }

    public function deactivate() {
        // Clear all cache on deactivation
        $cache_handler = new SEOINUX_Cache_Handler();
        $cache_handler->clear_all();

        flush_rewrite_rules();
    }
}

// Initialize the plugin
function seoinux_cache_pro_init() {
    return SEOINUX_Cache_Pro::get_instance();
}

// Start the plugin
seoinux_cache_pro_init();
